<?php

namespace App\Services;

use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FbrApiService
{
        /**
     * Get the base URL for production
     */
    private function getBaseUrl(): string
    {
        return config('fbr.production_url');
    }

    /**
     * Get the endpoint for production
     */
    private function getEndpoint(string $endpointKey): string
    {
        return config("fbr.endpoints.production.{$endpointKey}");
    }

    /**
     * Post invoice data to FBR
     */
    public function postInvoiceData(string $accessToken, array $invoiceData): array
    {
        try {
            $apiUrl = $this->getBaseUrl() . $this->getEndpoint('post_invoice');

            // Log the complete payload and API URL for debugging
            Log::info('=== FBR GENERATE INVOICE API CALL ===', [
                'api_url' => $apiUrl,
                'complete_payload' => $invoiceData,
                'payload_json' => json_encode($invoiceData, JSON_PRETTY_PRINT),
                'date_value' => $invoiceData['invoiceDate'] ?? 'Not set',
                'date_format_check' => 'Expected format: YYYY-MM-DD, Current: ' . ($invoiceData['invoiceDate'] ?? 'Not set')
            ]);

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json'
                ])->post($apiUrl, $invoiceData);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in postInvoiceData: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to post invoice data: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Validate invoice data with FBR
     */
    public function validateInvoiceData(string $accessToken, array $invoiceData): array
    {
        try {
            $apiUrl = $this->getBaseUrl() . $this->getEndpoint('validate_invoice');

            // Log the complete payload and API URL for debugging
            Log::info('=== FBR VALIDATE INVOICE API CALL ===', [
                'api_url' => $apiUrl,
                'complete_payload' => $invoiceData,
                'payload_json' => json_encode($invoiceData, JSON_PRETTY_PRINT),
                'date_value' => $invoiceData['invoiceDate'] ?? 'Not set',
                'date_format_check' => 'Expected format: YYYY-MM-DD, Current: ' . ($invoiceData['invoiceDate'] ?? 'Not set'),
                'payload_structure_check' => [
                    'has_flat_structure' => !isset($invoiceData['seller']) && !isset($invoiceData['buyer']),
                    'has_sellerNTNCNIC' => isset($invoiceData['sellerNTNCNIC']),
                    'has_buyerNTNCNIC' => isset($invoiceData['buyerNTNCNIC']),
                    'items_count' => count($invoiceData['items'] ?? [])
                ]
            ]);

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json'
                ])->post($apiUrl, $invoiceData);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in validateInvoiceData: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to validate invoice data: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get province codes
     */
    public function getProvinceCodes(string $accessToken): array
    {
        try {
            // Provinces use a different base URL than other endpoints
            $provincesUrl = 'https://gw.fbr.gov.pk/pdi/v1/provinces';

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($provincesUrl);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getProvinceCodes: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch province codes: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get document type IDs
     */
    public function getDocumentTypeIds(string $accessToken): array
    {
        try {
            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($this->getBaseUrl() . $this->getEndpoint('document_types'));

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getDocumentTypeIds: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch document type IDs: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get item codes
     */
    public function getItemCodes(string $accessToken): array
    {
        try {
            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($this->getBaseUrl() . $this->getEndpoint('item_codes'));

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getItemCodes: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch item codes: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get transaction type codes
     */
    public function getTransactionTypeCodes(string $accessToken): array
    {
        try {
            // Transaction type codes use PDI endpoint
            $transactionTypesUrl = 'https://gw.fbr.gov.pk/pdi/v1/transtypecode';

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($transactionTypesUrl);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getTransactionTypeCodes: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch transaction type codes: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get units of measurement
     */
    public function getUnitsOfMeasurement(string $accessToken): array
    {
        try {
            // UoM uses PDI endpoint as specified in user requirements
            $uomUrl = 'https://gw.fbr.gov.pk/pdi/v1/uom';

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($uomUrl);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getUnitsOfMeasurement: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch units of measurement: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get SRO schedules
     */
    public function getSroSchedules(string $accessToken): array
    {
        try {
            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($this->getBaseUrl() . $this->getEndpoint('sro_schedules'));

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getSroSchedules: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch SRO schedules: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get tax rates
     */
    public function getTaxRates(string $accessToken): array
    {
        try {
            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($this->getBaseUrl() . $this->getEndpoint('tax_rates'));

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getTaxRates: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch tax rates: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get HS codes with UOM
     */
    public function getHsCodesWithUom(string $accessToken): array
    {
        try {
            // HS codes use PDI endpoint similar to provinces
            $hsCodesUrl = 'https://gw.fbr.gov.pk/pdi/v1/hscodewituom';

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($hsCodesUrl);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getHsCodesWithUom: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch HS codes with UOM: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get item description codes (HS Codes)
     */
    public function getItemDescriptionCodes(string $accessToken): array
    {
        try {
            // Item description codes endpoint
            $itemDescCodesUrl = 'https://gw.fbr.gov.pk/pdi/v1/itemdesccode';

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($itemDescCodesUrl);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getItemDescriptionCodes: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch item description codes: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get SRO item IDs
     */
    public function getSroItemIds(string $accessToken): array
    {
        try {
            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($this->getBaseUrl() . $this->getEndpoint('sro_item_ids'));

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getSroItemIds: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch SRO item IDs: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get registration type for a given registration number
     */
    public function getRegistrationType(string $accessToken, string $registrationNumber): array
    {
        try {
            // Registration type uses a different URL structure
            $registrationTypeUrl = 'https://gw.fbr.gov.pk/dist/v1/Get_Reg_Type';

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json'
                ])->post($registrationTypeUrl, [
                    'Registration_No' => $registrationNumber
                ]);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getRegistrationType: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch registration type: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get tax rate based on date, transaction type, and province
     */
    public function getSaleTypeToRate(string $accessToken, string $date, int $transTypeId, int $originationSupplier): array
    {
        try {
            // Format the URL with query parameters
            $rateUrl = 'https://gw.fbr.gov.pk/pdi/v2/SaleTypeToRate';
            $queryParams = [
                'date' => $date,
                'transTypeId' => $transTypeId,
                'originationSupplier' => $originationSupplier
            ];

            $response = Http::timeout(config('fbr.timeout', 30))
                ->connectTimeout(config('fbr.connect_timeout', 10))
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get($rateUrl, $queryParams);

            return $this->handleResponse($response);
        } catch (Exception $e) {
            Log::error('FBR API Error in getSaleTypeToRate: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch sale type to rate: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Handle HTTP response from FBR API
     */
    private function handleResponse($response): array
    {
        $statusCode = $response->status();
        $responseData = $response->json();

        switch ($statusCode) {
            case 200:
                return [
                    'success' => true,
                    'data' => $responseData,
                    'status_code' => $statusCode
                ];

            case 401:
                return [
                    'success' => false,
                    'message' => 'Unauthorized - Invalid or expired FBR access token.',
                    'status_code' => $statusCode
                ];

            case 400:
                return [
                    'success' => false,
                    'message' => 'Bad Request - ' . ($responseData['message'] ?? 'Invalid request data'),
                    'status_code' => $statusCode,
                    'errors' => $responseData['errors'] ?? null
                ];

            case 500:
                return [
                    'success' => false,
                    'message' => 'Internal Server Error - FBR server issue.',
                    'status_code' => $statusCode
                ];

            default:
                return [
                    'success' => false,
                    'message' => 'Unexpected API response',
                    'status_code' => $statusCode,
                    'response' => $responseData
                ];
        }
    }
}
