<?php

namespace App\Http\Controllers;

use App\Services\FbrApiService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class InvoicingController extends Controller
{
    private $fbrApiService;

    public function __construct(FbrApiService $fbrApiService)
    {
        $this->middleware('auth');
        $this->fbrApiService = $fbrApiService;
    }

    /**
     * Display the invoicing form
     */
    public function index()
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return redirect()->route('profile.edit')
                ->with('warning', 'Please set your FBR Access Token in your profile to use the invoicing system.');
        }

        // Load data from FBR API
        $provinces = [];
        $hsCodes = [];
        $uoMs = [];
        $transactionTypes = [];

        try {
            // Load provinces
            $provincesResult = $this->fbrApiService->getProvinceCodes($user->fbr_access_token);
            if ($provincesResult['success']) {
                $provinces = $provincesResult['data'] ?? [];
            }

            // Load HS codes (Item Description Codes)
            $hsCodesResult = $this->fbrApiService->getItemDescriptionCodes($user->fbr_access_token);
            if ($hsCodesResult['success']) {
                $hsCodes = $hsCodesResult['data'] ?? [];
            }

            // Load Units of Measurement
            $uoMsResult = $this->fbrApiService->getUnitsOfMeasurement($user->fbr_access_token);
            if ($uoMsResult['success']) {
                $uoMs = $uoMsResult['data'] ?? [];
            }

            // Load Transaction Types
            $transactionTypesResult = $this->fbrApiService->getTransactionTypeCodes($user->fbr_access_token);
            if ($transactionTypesResult['success']) {
                $transactionTypes = $transactionTypesResult['data'] ?? [];
            }
        } catch (\Exception $e) {
            Log::error('Error loading FBR data', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
        }

        return view('invoicing.index', compact('provinces', 'hsCodes', 'uoMs', 'transactionTypes'));
    }

    /**
     * Submit invoice data to FBR
     */
    public function submitInvoice(Request $request)
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required. Please update your profile.'
            ], 400);
        }

        // Validate the request
        $validator = Validator::make($request->all(), [
            'invoiceType' => 'required|string',
            'invoiceDate' => 'required|date',
            'sellerNTNCNIC' => 'required|string|max:20',
            'sellerBusinessName' => 'required|string|max:255',
            'sellerAddress' => 'required|string',
            'sellerProvince' => 'required|string|max:100',
            'buyerNTNCNIC' => 'required|string|max:20',
            'buyerBusinessName' => 'required|string|max:255',
            'buyerProvince' => 'required|string|max:100',
            'buyerAddress' => 'required|string',
            'buyerRegistrationType' => 'required|string',
            'scenarioId' => 'required|string',
            'items' => 'required|array|min:1',
            'items.*.hsCode' => 'required|string',
            'items.*.productDescription' => 'required|string',
            'items.*.rate' => 'required|numeric|min:0',
            'items.*.uoM' => 'required|string',
            'items.*.quantity' => 'required|numeric|min:0',
            'items.*.totalValues' => 'required|numeric|min:0',
            'items.*.valueSalesExcludingST' => 'required|numeric|min:0',
            'items.*.salesTaxApplicable' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Load reference data for mapping
            $referenceData = $this->loadReferenceData($user->fbr_access_token);

            // Prepare invoice data according to FBR API specifications
            $invoiceData = $this->prepareInvoiceData($request->all(), $referenceData);

            // Submit to FBR
            $result = $this->fbrApiService->postInvoiceData(
                $user->fbr_access_token,
                $invoiceData
            );

            if ($result['success']) {
                Log::info('Invoice submitted successfully', [
                    'user_id' => $user->id,
                    'invoice_data' => $invoiceData,
                    'fbr_response' => $result['data']
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Invoice submitted successfully to FBR',
                    'data' => $result['data']
                ]);
            } else {
                Log::warning('Invoice submission failed', [
                    'user_id' => $user->id,
                    'error' => $result['message'],
                    'invoice_data' => $invoiceData
                ]);

                return response()->json([
                    'success' => false,
                    'message' => $result['message'],
                    'errors' => $result['errors'] ?? null
                ], $result['status_code'] ?? 400);
            }

        } catch (\Exception $e) {
            Log::error('Invoice submission exception', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while submitting the invoice: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Validate invoice data with FBR
     */
    public function validateInvoice(Request $request)
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        try {
            // Load reference data for mapping
            $referenceData = $this->loadReferenceData($user->fbr_access_token);

            $invoiceData = $this->prepareInvoiceData($request->all(), $referenceData);

            // Log comparison with expected payload structure
            Log::info('=== PAYLOAD STRUCTURE COMPARISON ===', [
                'current_payload' => $invoiceData,
                'expected_keys' => [
                    'invoiceType', 'invoiceDate', 'sellerNTNCNIC', 'sellerBusinessName',
                    'sellerProvince', 'sellerAddress', 'buyerNTNCNIC', 'buyerBusinessName',
                    'buyerProvince', 'buyerAddress', 'buyerRegistrationType', 'invoiceRefNo',
                    'scenarioId', 'items'
                ],
                'current_keys' => array_keys($invoiceData),
                'sample_item' => $invoiceData['items'][0] ?? null,
                'expected_item_keys' => [
                    'hsCode', 'productDescription', 'rate', 'uoM', 'quantity', 'totalValues',
                    'valueSalesExcludingST', 'fixedNotifiedValueOrRetailPrice', 'salesTaxApplicable',
                    'salesTaxWithheldAtSource', 'extraTax', 'furtherTax', 'sroScheduleNo',
                    'fedPayable', 'discount', 'saleType', 'sroItemSerialNo'
                ]
            ]);

            $result = $this->fbrApiService->validateInvoiceData(
                $user->fbr_access_token,
                $invoiceData
            );

            return response()->json($result);

        } catch (\Exception $e) {
            Log::error('Invoice validation exception', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Validation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get province codes from FBR API
     */
    public function getProvinceCodes()
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        $result = $this->fbrApiService->getProvinceCodes($user->fbr_access_token);
        return response()->json($result);
    }

    /**
     * Get HS codes with UOM from FBR API
     */
    public function getHsCodes()
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        try {
            $result = $this->fbrApiService->getHsCodesWithUom($user->fbr_access_token);
            return response()->json($result);
        } catch (\Exception $e) {
            Log::error('Error fetching HS codes', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch HS codes: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get item description codes from FBR API
     */
    public function getItemDescriptionCodes()
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        try {
            $result = $this->fbrApiService->getItemDescriptionCodes($user->fbr_access_token);
            return response()->json($result);
        } catch (\Exception $e) {
            Log::error('Error fetching item description codes', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch item description codes: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get units of measurement from FBR API
     */
    public function getUnitsOfMeasurement()
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        $result = $this->fbrApiService->getUnitsOfMeasurement($user->fbr_access_token);
        return response()->json($result);
    }

    /**
     * Get transaction type codes from FBR API
     */
    public function getTransactionTypeCodes()
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        $result = $this->fbrApiService->getTransactionTypeCodes($user->fbr_access_token);
        return response()->json($result);
    }

    /**
     * Get tax rates from FBR API
     */
    public function getTaxRates()
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        try {
            $result = $this->fbrApiService->getTaxRates($user->fbr_access_token);
            return response()->json($result);
        } catch (\Exception $e) {
            Log::error('Error fetching tax rates', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch tax rates: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get registration type for a given NTN/CNIC
     */
    public function getRegistrationType(Request $request)
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'registration_no' => 'required|string|max:20'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid registration number.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $result = $this->fbrApiService->getRegistrationType(
                $user->fbr_access_token,
                $request->registration_no
            );

            // Handle FBR specific response format
            if ($result['success'] && isset($result['data'])) {
                $data = $result['data'];

                // Check if the response has the expected structure
                if (isset($data['statuscode'])) {
                    if ($data['statuscode'] === '00' || $data['statuscode'] === '01') {
                        return response()->json([
                            'success' => true,
                            'data' => [
                                'registration_no' => $data['REGISTRATION_NO'] ?? $request->registration_no,
                                'registration_type' => $data['REGISTRATION_TYPE'] ?? 'unregistered'
                            ]
                        ]);
                    } else {
                        return response()->json([
                            'success' => false,
                            'message' => 'Invalid registration number or API error.'
                        ], 400);
                    }
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'Unexpected response format from FBR API.'
                    ], 500);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result['message'] ?? 'Failed to fetch registration type.'
                ], 500);
            }

        } catch (\Exception $e) {
            Log::error('Error fetching registration type', [
                'user_id' => $user->id,
                'registration_no' => $request->registration_no,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch registration type: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get sale type to rate based on date, transaction type, and province
     */
    public function getSaleTypeToRate(Request $request)
    {
        $user = Auth::user();

        if (!$user->fbr_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'FBR Access Token is required.'
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'date' => 'required|date',
            'trans_type_id' => 'required|integer',
            'origination_supplier' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid request parameters.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Format date to DD-MmmYYYY format as expected by FBR API (e.g., 24-Feb2024)
            $date = \Carbon\Carbon::parse($request->date)->format('d-MY');

            $result = $this->fbrApiService->getSaleTypeToRate(
                $user->fbr_access_token,
                $date,
                $request->trans_type_id,
                $request->origination_supplier
            );

            if ($result['success']) {
                return response()->json([
                    'success' => true,
                    'data' => $result['data']
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result['message'] ?? 'Failed to fetch rate information.'
                ], 500);
            }

        } catch (\Exception $e) {
            Log::error('Error fetching sale type to rate', [
                'user_id' => $user->id,
                'date' => $request->date,
                'trans_type_id' => $request->trans_type_id,
                'origination_supplier' => $request->origination_supplier,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch rate information: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Prepare invoice data according to FBR API specifications
     */
    private function prepareInvoiceData(array $requestData, array $referenceData): array
    {
        // Helper function to find province description by code
        $getProvinceDesc = function($code) use ($referenceData) {
            $provincesData = $referenceData['provinces'] ?? [];

            Log::info('=== PROVINCE MAPPING DEBUG ===', [
                'searching_for_code' => $code,
                'code_type' => gettype($code),
                'provinces_count' => count($provincesData),
                'provinces_sample' => array_slice($provincesData, 0, 3)
            ]);

            foreach ($provincesData as $province) {
                $provinceCode = $province['stateProvinceCode'] ?? '';
                $provinceDesc = $province['stateProvinceDesc'] ?? '';

                // Try both string and numeric comparison
                if ($provinceCode == $code || $provinceCode === (string)$code || $provinceCode === (int)$code) {
                    Log::info('Province mapping found', [
                        'code' => $code,
                        'matched_province_code' => $provinceCode,
                        'description' => $provinceDesc
                    ]);
                    return $provinceDesc ?: $code;
                }
            }

            Log::warning('Province mapping not found', [
                'code' => $code,
                'returning_original' => $code
            ]);
            return $code; // Return original if not found
        };

        // Helper function to find UoM description by ID
        $getUoMDesc = function($id) use ($referenceData) {
            $uoMsData = $referenceData['uoMs'] ?? [];

            Log::info('=== UOM MAPPING DEBUG ===', [
                'searching_for_id' => $id,
                'id_type' => gettype($id),
                'uoms_count' => count($uoMsData),
                'uoms_sample' => array_slice($uoMsData, 0, 3)
            ]);

            foreach ($uoMsData as $uom) {
                $uomId = $uom['uoM_ID'] ?? $uom['id'] ?? '';
                $uomDesc = $uom['uoM_DESC'] ?? $uom['description'] ?? '';

                // Try both string and numeric comparison
                if ($uomId == $id || $uomId === (string)$id || $uomId === (int)$id) {
                    Log::info('UoM mapping found', [
                        'id' => $id,
                        'matched_uom_id' => $uomId,
                        'description' => $uomDesc
                    ]);
                    return $uomDesc ?: $id;
                }
            }

            Log::warning('UoM mapping not found', [
                'id' => $id,
                'returning_original' => $id
            ]);
            return $id; // Return original if not found
        };

        // Helper function to find transaction type description by ID
        $getTransactionTypeDesc = function($id) use ($referenceData) {
            $transactionTypesData = $referenceData['transactionTypes'] ?? [];

            Log::info('=== TRANSACTION TYPE MAPPING DEBUG ===', [
                'searching_for_id' => $id,
                'id_type' => gettype($id),
                'transaction_types_count' => count($transactionTypesData),
                'transaction_types_sample' => array_slice($transactionTypesData, 0, 3)
            ]);

            foreach ($transactionTypesData as $type) {
                $typeId = $type['transactioN_TYPE_ID'] ?? '';
                $typeDesc = $type['transactioN_DESC'] ?? '';

                // Try both string and numeric comparison
                if ($typeId == $id || $typeId === (string)$id || $typeId === (int)$id) {
                    Log::info('Transaction type mapping found', [
                        'id' => $id,
                        'matched_type_id' => $typeId,
                        'description' => $typeDesc
                    ]);
                    return $typeDesc ?: $id;
                }
            }

            Log::warning('Transaction type mapping not found', [
                'id' => $id,
                'returning_original' => $id
            ]);
            return $id; // Return original if not found
        };

        // Prepare items array
        $items = [];
        foreach ($requestData['items'] as $item) {
            // Ensure rate has % symbol
            $rate = $item['rate'] ?? '';
            if (!empty($rate) && !str_contains($rate, '%')) {
                $rate = $rate . '%';
            }

            // Convert UoM ID to description
            $uoMLabel = $getUoMDesc($item['uoM'] ?? '');

            // Convert sale type ID to description
            $saleTypeLabel = $getTransactionTypeDesc($item['saleType'] ?? '');

            $items[] = [
                'hsCode' => $item['hsCode'],
                'productDescription' => $item['productDescription'],
                'rate' => $rate, // String value with % sign
                'uoM' => $uoMLabel, // Use description instead of ID
                'quantity' => (int) $item['quantity'],
                'totalValues' => (int) $item['totalValues'],
                'valueSalesExcludingST' => (int) $item['valueSalesExcludingST'],
                'fixedNotifiedValueOrRetailPrice' => (int) ($item['fixedNotifiedValueOrRetailPrice'] ?? 0),
                'salesTaxApplicable' => (int) $item['salesTaxApplicable'],
                'salesTaxWithheldAtSource' => (int) ($item['salesTaxWithheldAtSource'] ?? 0),
                'extraTax' => $item['extraTax'] ?? '',
                'furtherTax' => (int) ($item['furtherTax'] ?? 0),
                'sroScheduleNo' => $item['sroScheduleNo'] ?? '',
                'fedPayable' => (int) ($item['fedPayable'] ?? 0),
                'discount' => (int) ($item['discount'] ?? 0),
                'saleType' => $saleTypeLabel, // Use description instead of ID
                'sroItemSerialNo' => $item['sroItemSerialNo'] ?? ''
            ];
        }

        // Convert province codes to descriptions
        $sellerProvinceLabel = $getProvinceDesc($requestData['sellerProvince'] ?? '');
        $buyerProvinceLabel = $getProvinceDesc($requestData['buyerProvince'] ?? '');

        // Create flat structure to match expected payload
        $invoiceData = [
            'invoiceType' => $requestData['invoiceType'],
            'invoiceDate' => $requestData['invoiceDate'],
            'sellerNTNCNIC' => $requestData['sellerNTNCNIC'],
            'sellerBusinessName' => $requestData['sellerBusinessName'],
            'sellerProvince' => $sellerProvinceLabel, // Use description instead of code
            'sellerAddress' => $requestData['sellerAddress'],
            'buyerNTNCNIC' => $requestData['buyerNTNCNIC'],
            'buyerBusinessName' => $requestData['buyerBusinessName'],
            'buyerProvince' => $buyerProvinceLabel, // Use description instead of code
            'buyerAddress' => $requestData['buyerAddress'],
            'buyerRegistrationType' => $requestData['buyerRegistrationType'],
            'invoiceRefNo' => $requestData['invoiceRefNo'] ?? '',
            'scenarioId' => $requestData['scenarioId'],
            'items' => $items
        ];

        // Log the payload for debugging with ID to label mappings
        Log::info('Prepared Invoice Data Payload with Label Mappings', [
            'payload' => $invoiceData,
            'date_format' => $requestData['invoiceDate'],
            'items_count' => count($items),
            'label_mappings' => [
                'seller_province' => $requestData['sellerProvince'] . ' -> ' . $sellerProvinceLabel,
                'buyer_province' => $requestData['buyerProvince'] . ' -> ' . $buyerProvinceLabel,
                'sample_uom_mapping' => isset($items[0]) ? ($requestData['items'][0]['uoM'] ?? '') . ' -> ' . ($items[0]['uoM'] ?? '') : 'No items',
                'sample_saletype_mapping' => isset($items[0]) ? ($requestData['items'][0]['saleType'] ?? '') . ' -> ' . ($items[0]['saleType'] ?? '') : 'No items'
            ]
        ]);

        return $invoiceData;
    }

    /**
     * Load reference data for mapping
     */
    private function loadReferenceData($accessToken)
    {
        $provinces = [];
        $hsCodes = [];
        $uoMs = [];
        $transactionTypes = [];

        try {
            // Load provinces
            $provincesResult = $this->fbrApiService->getProvinceCodes($accessToken);
            if ($provincesResult['success']) {
                $provinces = $provincesResult['data'] ?? [];
            }

            // Load HS codes (Item Description Codes)
            $hsCodesResult = $this->fbrApiService->getItemDescriptionCodes($accessToken);
            if ($hsCodesResult['success']) {
                $hsCodes = $hsCodesResult['data'] ?? [];
            }

            // Load Units of Measurement
            $uoMsResult = $this->fbrApiService->getUnitsOfMeasurement($accessToken);
            if ($uoMsResult['success']) {
                $uoMs = $uoMsResult['data'] ?? [];
            }

            // Load Transaction Types
            $transactionTypesResult = $this->fbrApiService->getTransactionTypeCodes($accessToken);
            if ($transactionTypesResult['success']) {
                $transactionTypes = $transactionTypesResult['data'] ?? [];
            }

            // Debug log the loaded reference data
            Log::info('=== REFERENCE DATA LOADED ===', [
                'provinces_count' => count($provinces),
                'provinces_sample' => array_slice($provinces, 0, 2),
                'uoms_count' => count($uoMs),
                'uoms_sample' => array_slice($uoMs, 0, 2),
                'transaction_types_count' => count($transactionTypes),
                'transaction_types_sample' => array_slice($transactionTypes, 0, 2)
            ]);

        } catch (\Exception $e) {
            Log::error('Error loading reference data', [
                'error' => $e->getMessage()
            ]);
        }

        return [
            'provinces' => $provinces,
            'hsCodes' => $hsCodes,
            'uoMs' => $uoMs,
            'transactionTypes' => $transactionTypes
        ];
    }
}
